<?php

namespace App\Http\Controllers;

use App\Models\Router;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Services\Router\MikroTikHotspotService;

class HotspotVoucherController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $routers = Router::orderBy('name')->get();
        $routerId = $request->query('router_id') ?: $routers->first()?->id;
        $router = $routers->firstWhere('id', $routerId);

        $users = [];
        if ($router) {
            $users = MikroTikHotspotService::listUsers($router);
        }

        return view('hotspot.vouchers.index', compact('routers','routerId','router','users'));
    }

    public function create(Request $request)
    {
        $routers = Router::orderBy('name')->get();
        $routerId = $request->query('router_id') ?: $routers->first()?->id;
        $router = $routers->firstWhere('id', $routerId);

        $profiles = $router ? MikroTikHotspotService::listProfiles($router) : [];

        return view('hotspot.vouchers.create', compact('routers','routerId','router','profiles'));
    }

    public function store(Request $r)
    {
        $data = $r->validate([
            'router_id'     => 'required|exists:routers,id',
            'profile'       => 'required|string',
            'username'      => 'nullable|string|max:191',
            'password'      => 'nullable|string|max:191',
            'limit_uptime'  => 'nullable|string|max:50',
            'comment'       => 'nullable|string|max:191',
        ]);

        $router = Router::findOrFail($data['router_id']);

        $username = $data['username'] ?: strtoupper(Str::random(8));
        $password = $data['password'] ?: Str::random(6);

        MikroTikHotspotService::createUser($router, [
            'name'         => $username,
            'password'     => $password,
            'profile'      => $data['profile'],
            'limit_uptime' => $data['limit_uptime'] ?? null,
            'comment'      => $data['comment'] ?? 'BlueBando v3 single voucher',
        ]);

        // TODO: optionally insert into vouchers table for reporting

        return redirect()
            ->route('hotspot.vouchers.index',['router_id'=>$router->id])
            ->with('status', "Voucher {$username} created.");
    }

    public function createBulk(Request $request)
    {
        $routers = Router::orderBy('name')->get();
        $routerId = $request->query('router_id') ?: $routers->first()?->id;
        $router = $routers->firstWhere('id', $routerId);
        $profiles = $router ? MikroTikHotspotService::listProfiles($router) : [];

        return view('hotspot.vouchers.bulk', compact('routers','routerId','router','profiles'));
    }

    public function storeBulk(Request $r)
    {
        $data = $r->validate([
            'router_id'     => 'required|exists:routers,id',
            'profile'       => 'required|string',
            'quantity'      => 'required|integer|min:1|max:500',
            'limit_uptime'  => 'nullable|string|max:50',
            'comment'       => 'nullable|string|max:191',
        ]);

        $router = Router::findOrFail($data['router_id']);

        $created = [];
        for ($i=0; $i < $data['quantity']; $i++) {
            $user = strtoupper(Str::random(8));
            $pass = Str::random(6);

            MikroTikHotspotService::createUser($router, [
                'name'         => $user,
                'password'     => $pass,
                'profile'      => $data['profile'],
                'limit_uptime' => $data['limit_uptime'] ?? null,
                'comment'      => $data['comment'] ?? 'BlueBando v3 bulk voucher',
            ]);

            $created[] = ['username'=>$user,'password'=>$pass];
        }

        // We can pass them to a simple print view
        return view('hotspot.vouchers.print-bulk', compact('router','created'))
            ->with('status', 'Bulk vouchers generated.');
    }

    public function destroy(Request $r, Router $router, string $username)
    {
        $users = MikroTikHotspotService::listUsers($router);
        $user = collect($users)->firstWhere('name', $username);
        if ($user && !empty($user['id'])) {
            MikroTikHotspotService::deleteUser($router, $user['id']);
        }

        return redirect()
            ->route('hotspot.vouchers.index',['router_id'=>$router->id])
            ->with('status',"Voucher {$username} deleted.");
    }

    public function printSingle(Router $router, string $username)
    {
        $users = MikroTikHotspotService::listUsers($router);
        $user = collect($users)->firstWhere('name', $username);
        if (!$user) {
            abort(404);
        }

        return view('hotspot.vouchers.print-single', compact('router','user'));
    }
}
