<?php

namespace App\Http\Controllers;

use App\Models\Freelancer;
use App\Models\Router;
use App\Models\Order;
use App\Models\User;
use Illuminate\Http\Request;

class FreelancerController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    // List + small stats
    public function index(Request $request)
    {
        $status = $request->input('status');

        $q = Freelancer::query();

        if ($status) {
            $q->where('status', $status);
        }

        if ($search = $request->input('q')) {
            $q->where(function ($x) use ($search) {
                $x->where('name', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%")
                  ->orWhere('code', 'like', "%{$search}%");
            });
        }

        $freelancers = $q->orderBy('name')->paginate(20);

        // Global stats
        $stats = [
            'count'        => Freelancer::count(),
            'active'       => Freelancer::where('status', 'active')->count(),
            'suspended'    => Freelancer::where('status', 'suspended')->count(),
            'month_sales'  => Order::whereNotNull('freelancer_id')
                ->where('status', 'success')
                ->whereMonth('created_at', now()->month)
                ->sum('amount'),
        ];

        return view('freelancers.index', compact('freelancers', 'status', 'stats'));
    }

    public function create()
    {
        $routers = Router::orderBy('name')->get();
        $users   = User::orderBy('name')->get();

        return view('freelancers.create', compact('routers', 'users'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name'            => ['required','string','max:191'],
            'phone'           => ['required','string','max:50'],
            'email'           => ['nullable','email','max:191'],
            'code'            => ['nullable','string','max:50','unique:freelancers,code'],
            'commission_rate' => ['required','numeric','min:0','max:100'],
            'status'          => ['required','in:active,suspended'],
            'notes'           => ['nullable','string'],
            'user_id'         => ['nullable','integer','exists:users,id'],
            'router_ids'      => ['array'],
            'router_ids.*'    => ['integer','exists:routers,id'],
        ]);

        $data['company_id'] = 1;

        $freelancer = Freelancer::create($data);

        if (!empty($data['router_ids'])) {
            $freelancer->routers()->sync($data['router_ids']);
        }

        return redirect()->route('freelancers.index')
            ->with('status', 'Freelancer created.');
    }

    public function show(Freelancer $freelancer)
    {
        $freelancer->load('routers');

        $orders = Order::where('freelancer_id', $freelancer->id)
            ->where('status', 'success')
            ->orderByDesc('id')
            ->limit(20)
            ->get();

        $totalSales = $orders->sum('amount');
        $commission = round($totalSales * ($freelancer->commission_rate / 100), 2);

        return view('freelancers.show', [
            'freelancer' => $freelancer,
            'orders'     => $orders,
            'totalSales' => $totalSales,
            'commission' => $commission,
        ]);
    }

    public function edit(Freelancer $freelancer)
    {
        $routers   = Router::orderBy('name')->get();
        $assigned  = $freelancer->routers()->pluck('router_id')->toArray();
        $users     = User::orderBy('name')->get();

        return view('freelancers.edit', compact('freelancer','routers','assigned','users'));
    }

    public function update(Request $request, Freelancer $freelancer)
    {
        $data = $request->validate([
            'name'            => ['required','string','max:191'],
            'phone'           => ['required','string','max:50'],
            'email'           => ['nullable','email','max:191'],
            'code'            => ['nullable','string','max:50','unique:freelancers,code,' . $freelancer->id],
            'commission_rate' => ['required','numeric','min:0','max:100'],
            'status'          => ['required','in:active,suspended'],
            'notes'           => ['nullable','string'],
            'user_id'         => ['nullable','integer','exists:users,id'],
            'router_ids'      => ['array'],
            'router_ids.*'    => ['integer','exists:routers,id'],
        ]);

        $freelancer->update($data);

        if (!empty($data['router_ids'])) {
            $freelancer->routers()->sync($data['router_ids']);
        } else {
            $freelancer->routers()->detach();
        }

        return redirect()->route('freelancers.index')
            ->with('status', 'Freelancer updated.');
    }

    public function destroy(Freelancer $freelancer)
    {
        $freelancer->routers()->detach();
        $freelancer->delete();

        return redirect()->route('freelancers.index')
            ->with('status', 'Freelancer deleted.');
    }
}
