<?php

namespace App\Services\Router;

use App\Models\Plan;
use App\Models\Router;
use RouterOS\Client;
use RouterOS\Query;

class MikroTikPlanService
{
    /**
     * Get a RouterOS client using our existing MikroTikLive helper.
     * Adjust this if your helper name/method is different.
     */
    protected static function client(Router $router): ?Client
    {
        // Example: MikroTikLive::client($router)
        // If your helper is App\Services\Router\MikroTikLive with connect($router),
        // change the line below accordingly.
        return MikroTikLive::client($router);
    }

    /**
     * Ensure a hotspot user profile exists/updated for this plan on its router.
     */
    public static function syncPlanToRouter(Plan $plan): void
    {
        if (!$plan->router_id) {
            return;
        }

        $router = $plan->router;
        if (!$router) {
            return;
        }

        $client = static::client($router);
        if (!$client) {
            return;
        }

        $profileName = $plan->profile ?: $plan->name;

        // 1) Check if profile already exists
        $q = (new Query('/ip/hotspot/user/profile/print'))
            ->where('name', $profileName);

        $rows = $client->query($q)->read();

        $fields = [];
        if ($plan->rate_limit) {
            $fields['rate-limit'] = $plan->rate_limit;          // e.g. "5M/5M"
        }

        $fields['comment'] = 'BlueBando v3 • ' . ($plan->validity_hours ?? '-') . ' hrs';

        if (!empty($rows)) {
            // Update existing profile
            $id = $rows[0]['.id'] ?? null;
            if ($id) {
                $q = (new Query('/ip/hotspot/user/profile/set'))
                    ->equal('.id', $id);

                foreach ($fields as $k => $v) {
                    $q->equal($k, $v);
                }

                $client->query($q)->read();
            }
        } else {
            // Create new profile
            $q = (new Query('/ip/hotspot/user/profile/add'))
                ->equal('name', $profileName);

            foreach ($fields as $k => $v) {
                $q->equal($k, $v);
            }

            $client->query($q)->read();
        }
    }

    /**
     * Delete the hotspot profile for this plan from its router (if present).
     */
    public static function deletePlanFromRouter(Plan $plan): void
    {
        if (!$plan->router_id) {
            return;
        }

        $router = $plan->router;
        if (!$router) {
            return;
        }

        $client = static::client($router);
        if (!$client) {
            return;
        }

        $profileName = $plan->profile ?: $plan->name;

        $q = (new Query('/ip/hotspot/user/profile/print'))
            ->where('name', $profileName);

        $rows = $client->query($q)->read();

        if (!empty($rows)) {
            $id = $rows[0]['.id'] ?? null;
            if ($id) {
                $q = (new Query('/ip/hotspot/user/profile/remove'))
                    ->equal('.id', $id);

                $client->query($q)->read();
            }
        }
    }

    /**
     * Return all hotspot user profiles from a router (for listing on Plans page).
     *
     * @return array<int, array<string,mixed>>
     */
    public static function listProfilesForRouter(Router $router): array
    {
        $client = static::client($router);
        if (!$client) {
            return [];
        }

        $q = new Query('/ip/hotspot/user/profile/print');

        $rows = $client->query($q)->read();

        $profiles = [];
        foreach ($rows as $row) {
            $profiles[] = [
                'name'              => $row['name'] ?? '',
                'rate_limit'        => $row['rate-limit'] ?? null,
                'shared_users'      => $row['shared-users'] ?? null,
                'idle_timeout'      => $row['idle-timeout'] ?? null,
                'keepalive_timeout' => $row['keepalive-timeout'] ?? null,
                'parent_queue'      => $row['parent-queue'] ?? null,
                'comment'           => $row['comment'] ?? null,
            ];
        }

        return $profiles;
    }
}