<?php

namespace App\Services\Payments;

use Illuminate\Support\Facades\Http;

class AzamPayService
{
    protected string $baseUrl;
    protected string $appName;
    protected string $clientId;
    protected string $clientSecret;
    protected string $callbackUrl;

    public function __construct()
{
    $this->baseUrl  = rtrim(config('azam.base_url'), '/');
    $this->appName  = config('azam.app_name', 'BlueBando');

    $this->clientId = config('azam.client_id') ?? '';
    $this->clientSecret = config('azam.client_secret') ?? '';
    $this->accountNumber = config('azam.account_number') ?? '';

    if ($this->clientId === '' || $this->clientSecret === '') {
        throw new \Exception("AzamPay credentials missing. Please set AZAM_CLIENT_ID and AZAM_CLIENT_SECRET in .env");
    }
}


    /**
     * Minimal checkout for mobile money (non-card).
     */
    public function createOrderMinimal(
        string $reference,
        float  $amount,
        string $currency,
        string $phone,
        string $provider
    ): array {
        $token = $this->getToken();

        $url  = $this->baseUrl . '/v1/checkout/create-order-minimal';

        $payload = [
            "amount"        => (float) $amount,
            "currency"      => $currency,
            "externalId"    => $reference,
            "phoneNumber"   => $phone,
            "provider"      => $provider, // Airtel, Tigo, Halopesa, Azampesa
            "callbackUrl"   => $this->callbackUrl,
            "vendor"        => $this->appName,
            "description"   => "BlueBando WiFi payment",
        ];

        $response = Http::withToken($token)
            ->acceptJson()
            ->post($url, $payload);

        return [
            'request'  => $payload,
            'response' => $response->json(),
            'ok'       => $response->successful(),
            'status'   => $response->status(),
        ];
    }

    /**
     * Get OAuth token from AzamPay.
     */
   public function getToken()
{
    $url = $this->baseUrl . '/api/v1/payments/token';

    $payload = [
        'appName' => $this->appName,
        'clientId' => $this->clientId,
        'secret' => $this->clientSecret,
    ];

    $resp = Http::withoutVerifying()->post($url, $payload);

    if (!$resp->ok() || !isset($resp['data']['accessToken'])) {
        throw new \RuntimeException("Failed to get AzamPay token. Response: " . $resp->body());
    }

    return $resp['data']['accessToken'];
}

}
