<?php

namespace App\Http\Controllers;

use App\Models\Voucher;
use App\Models\Plan;
use App\Models\Router;
use App\Models\Payment;
use App\Services\Router\MikroTikProvisioner;
use App\Services\Sms\BeemSmsService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class VoucherController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index()
    {
        $vouchers = Voucher::with('router','plan','payment','creator')
            ->latest()->paginate(25);

        return view('vouchers.index', compact('vouchers'));
    }

    public function create()
    {
        $routers = Router::orderBy('name')->get();
        $plans   = Plan::where('is_active', true)->orderBy('price')->get();

        return view('vouchers.create', compact('routers','plans'));
    }

    public function store(Request $r)
    {
        $data = $r->validate([
            'router_id'   => 'required|exists:routers,id',
            'plan_id'     => 'required|exists:plans,id',
            'buyer_phone' => 'nullable|string|max:20',
            'send_sms'    => 'nullable|boolean',
        ]);

        $router = Router::findOrFail($data['router_id']);
        $plan   = Plan::findOrFail($data['plan_id']);

        // create internal "cash" payment (for reports/commissions)
        $payment = Payment::create([
            'gateway'         => 'manual',
            'reference'       => 'BB-CASH-'.Str::upper(Str::random(8)),
            'customer_msisdn' => $data['buyer_phone'] ?? null,
            'amount'          => $plan->price,
            'currency'        => $plan->currency,
            'status'          => 'success',
            'router_id'       => $router->id,
            'plan_id'         => $plan->id,
            'paid_at'         => now(),
        ]);

        // generate voucher credentials
        $code = strtoupper(Str::random(8));
        $pass = Str::random(6);

        $voucher = Voucher::create([
            'router_id'   => $router->id,
            'plan_id'     => $plan->id,
            'code'        => $code,
            'password'    => $pass,
            'status'      => 'active',
            'buyer_phone' => $data['buyer_phone'] ?? null,
            'price'       => $plan->price,
            'currency'    => $plan->currency,
            'payment_id'  => $payment->id,
            'created_by'  => auth()->id(),
        ]);

        // push to MikroTik
        try {
            MikroTikProvisioner::createHotspotUser($voucher);
        } catch (\Throwable $e) {
            $voucher->status = 'pending';
            $voucher->meta   = ['provision_error' => $e->getMessage()];
            $voucher->save();
        }

        // optional SMS
        if ($voucher->buyer_phone && $r->boolean('send_sms')) {
            $msg = "Thank you for using BlueBando.\n"
                 . "Voucher: {$voucher->code}\n"
                 . "Password: {$voucher->password}\n"
                 . "Plan: {$plan->name} ({$plan->validity_hours} hrs)";
            app(BeemSmsService::class)->send($voucher->buyer_phone, $msg);
        }

        return redirect()->route('vouchers.index')
            ->with('status','Voucher created.');
    }
}
