<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        // later you can add ->middleware('role:admin') if you like
    }

    public function index(Request $request)
    {
        $q = User::query();

        if ($search = $request->input('q')) {
            $q->where(function ($x) use ($search) {
                $x->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        $users = $q->orderBy('name')->paginate(20);

        $roles = Role::orderBy('name')->get();

        return view('users.index', compact('users', 'roles'));
    }

    public function create()
    {
        $roles = Role::orderBy('name')->get();
        return view('users.create', compact('roles'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name'     => ['required','string','max:191'],
            'email'    => ['required','email','max:191','unique:users,email'],
            'password' => ['required','string','min:6'],
            'roles'    => ['array'],
            'roles.*'  => ['string'],
        ]);

        $user = User::create([
            'name'     => $data['name'],
            'email'    => $data['email'],
            'password' => $data['password'],
        ]);

        if (!empty($data['roles'])) {
            $user->syncRoles($data['roles']);
        }

        return redirect()->route('users.index')
            ->with('status', 'User created.');
    }

    public function edit(User $user)
    {
        $roles = Role::orderBy('name')->get();
        $userRoles = $user->getRoleNames()->toArray();

        return view('users.edit', compact('user','roles','userRoles'));
    }

    public function update(Request $request, User $user)
    {
        $data = $request->validate([
            'name'     => ['required','string','max:191'],
            'email'    => ['required','email','max:191','unique:users,email,' . $user->id],
            'password' => ['nullable','string','min:6'],
            'roles'    => ['array'],
            'roles.*'  => ['string'],
        ]);

        $user->name  = $data['name'];
        $user->email = $data['email'];

        if (!empty($data['password'])) {
            $user->password = $data['password'];
        }

        $user->save();

        $user->syncRoles($data['roles'] ?? []);

        return redirect()->route('users.index')
            ->with('status', 'User updated.');
    }

    public function destroy(User $user)
    {
        if ($user->id === auth()->id()) {
            return back()->with('status', 'You cannot delete your own account.');
        }

        $user->delete();

        return redirect()->route('users.index')
            ->with('status', 'User deleted.');
    }
}
