<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Plan;
use App\Models\Router;
use App\Services\Payments\AzamPayService;
use App\Services\Router\MikroTikHotspotService;
use App\Services\Sms\BeemSms;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PaymentController extends Controller
{
    /**
     * Show public payment form.
     * URL: /payment.php?router_id=1
     */
    public function showForm(Request $request)
    {
        $routerId = $request->query('router_id');

        // If router_id missing, just take first active router
        $router = Router::query()
            ->when($routerId, fn ($q) => $q->where('id', $routerId))
            ->first();

        if (!$router) {
            abort(404, 'Router not found');
        }

        // Fetch plans linked to this router (and active)
        $plans = Plan::where('router_id', $router->id)
            ->where('is_active', true)
            ->orderBy('price')
            ->get();

        return view('payment.form', [
            'router' => $router,
            'plans'  => $plans,
        ]);
    }

    /**
     * Handle "Lipa Sasa" – create Order + call AzamPay.
     */
    public function createOrder(Request $request)
    {
        $routerId = $request->input('router_id');

        $router = Router::findOrFail($routerId);

        $request->validate([
            'network'   => 'required|string|in:Airtel,Tigo,Halopesa,Azampesa',
            'phone'     => 'required|string',
            'plan_id'   => 'required|exists:plans,id',
            'gateway'   => 'required|in:azampay',
        ]);

        $plan = Plan::where('id', $request->plan_id)
            ->where('router_id', $router->id)
            ->where('is_active', true)
            ->firstOrFail();

        // Normalize phone -> 2557xxxxxx
        $phone = $this->normalizeMsisdn($request->phone);

        // Create local order (PENDING)
        $reference = 'BB' . now()->format('YmdHis') . Str::random(4);

        $order = new Order();
        $order->router_id   = $router->id;
        $order->plan_id     = $plan->id;
        $order->phone       = $phone;
        $order->network     = $request->network;
        $order->gateway     = 'azampay';
        $order->amount      = $plan->price;
        $order->currency    = $plan->currency ?? 'TZS';
        $order->reference   = $reference;
        $order->status      = 'pending';
        $order->voucher_code = null;
        $order->meta        = [];
        $order->save();

        // Call AzamPay – minimal checkout
        $azam = new AzamPayService();
        $res  = $azam->createOrderMinimal(
            reference: $reference,
            amount: $plan->price,
            currency: $order->currency,
            phone: $phone,
            provider: $request->network,
        );

        // Save raw response meta
        $meta = $order->meta ?? [];
        $meta['azam_request']  = $res['request'] ?? null;
        $meta['azam_response'] = $res['response'] ?? null;
        $order->meta = $meta;
        $order->save();

        // Show instruction page (USSD / push)
        return view('payment.status', [
            'router'   => $router,
            'plan'     => $plan,
            'order'    => $order,
            'mode'     => 'pending',
            'azamInfo' => $res,
        ]);
    }

    /**
     * AzamPay callback → update order + generate voucher + send SMS.
     */
    public function azamCallback(Request $request)
    {
        // Adjust to exact Azam callback fields you receive
        $reference = $request->input('externalId'); // our reference
        $status    = strtolower($request->input('status', ''));
        $transId   = $request->input('transactionId');

        $order = Order::where('reference', $reference)->first();

        if (!$order) {
            return response()->json(['ok' => false, 'message' => 'Order not found'], 404);
        }

        $meta = $order->meta ?? [];
        $meta['azam_callback'] = $request->all();
        $order->meta = $meta;

        if ($status === 'success' || $status === 'successful') {

            if ($order->status !== 'success') {
                $order->status         = 'success';
                $order->transaction_id = $transId;

                // Generate voucher in MikroTik
                $router = $order->router;
                $plan   = $order->plan;

                if ($router && $plan) {
                    $voucher = MikroTikHotspotService::generateVoucher(
                        $router,
                        $plan,
                        $order->phone
                    );

                    $order->voucher_code = $voucher;

                    if ($voucher) {
                        // SMS text can be customized
                        $text = "Asante kwa malipo ya WiFi.\n"
                              . "Kifurushi: {$plan->name}\n"
                              . "Voucher: {$voucher}";
                        BeemSms::send($order->phone, $text);
                    }
                }
            }

        } else {
            $order->status = 'failed';
        }

        $order->save();

        return response()->json(['ok' => true]);
    }

    /**
     * Public result page (when user comes back).
     */
    public function showResult(string $reference)
    {
        $order = Order::where('reference', $reference)->firstOrFail();
        $router = $order->router;
        $plan   = $order->plan;

        $mode = $order->status; // pending/success/failed

        return view('payment.status', compact('order', 'router', 'plan', 'mode'));
    }

    /**
     * Normalize msisdn to 2557xxxxxxxx.
     */
    protected function normalizeMsisdn(string $phone): string
    {
        $p = preg_replace('/\D+/', '', $phone); // digits only

        if (str_starts_with($p, '0') && strlen($p) >= 10) {
            // 07xx → 2557xx
            $p = '255' . substr($p, 1);
        } elseif (!str_starts_with($p, '255')) {
            // anything else – just prefix 255
            $p = '255' . ltrim($p, '0');
        }

        return $p;
    }
}
