<?php

namespace App\Http\Controllers;

use App\Models\Router;
use Illuminate\Http\Request;
use App\Services\Router\MikroTikHotspotService;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Illuminate\Support\Facades\DB;

class HotspotActiveController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Active sessions page.
     */
    public function index(Request $request)
    {
        $routers = Router::orderBy('name')->get();
        if ($routers->isEmpty()) {
            return view('hotspot.active.index', [
                'sessions'      => [],
                'routers'       => $routers,
                'currentRouter' => null,
                'limit'         => 100,
                'offline'       => true,
                'sort'          => 'traffic',
                'dir'           => 'desc',
            ]);
        }

        $routerId = $request->integer('router_id') ?: $routers->first()->id;
        $limit    = $request->integer('limit', 100) ?: 100;
        $sort     = $request->get('sort', 'traffic'); // traffic|bytes_in|bytes_out|uptime|user
        $dir      = strtolower($request->get('dir', 'desc')) === 'asc' ? 'asc' : 'desc';

        $currentRouter = $routers->firstWhere('id', $routerId) ?? $routers->first();

        $offline  = false;
        $sessions = [];

        try {
            $sessions = MikroTikHotspotService::listActive($currentRouter, $limit);

            // Enrich rows with metrics
            foreach ($sessions as &$row) {
                $bi = (int)($row['bytes-in'] ?? $row['bytes_in'] ?? 0);
                $bo = (int)($row['bytes-out'] ?? $row['bytes_out'] ?? 0);
                $row['_bytes_in']  = $bi;
                $row['_bytes_out'] = $bo;
                $row['_traffic']   = $bi + $bo;

                // convert uptime to seconds for sorting
                $row['_uptime_sec'] = self::parseDurationToSeconds($row['uptime'] ?? '');
                $row['_user']       = $row['user'] ?? $row['name'] ?? '';
            }
            unset($row);

            // Sort
            $sessions = self::sortSessions($sessions, $sort, $dir);

            // Log snapshot for history
            $this->logSnapshot($currentRouter, $sessions);

        } catch (\Throwable $e) {
            $offline  = true;
            $sessions = [];
        }

        return view('hotspot.active.index', [
            'sessions'      => $sessions,
            'routers'       => $routers,
            'currentRouter' => $currentRouter,
            'limit'         => $limit,
            'offline'       => $offline,
            'sort'          => $sort,
            'dir'           => $dir,
        ]);
    }

    /**
     * Disconnect a single session.
     */
    public function kick(Request $request, Router $router)
    {
        $id = $request->input('id');

        if ($id) {
            try {
                MikroTikHotspotService::disconnectSession($router, $id);
                return back()->with('status', 'User disconnected successfully.');
            } catch (\Throwable $e) {
                return back()->with('error', 'Failed to disconnect user: '.$e->getMessage());
            }
        }

        return back()->with('error', 'Missing session id.');
    }

    /**
     * Disconnect all active sessions on this router.
     */
    public function kickAll(Request $request, Router $router)
    {
        try {
            MikroTikHotspotService::disconnectAll($router);
            return back()->with('status', 'All active users disconnected.');
        } catch (\Throwable $e) {
            return back()->with('error', 'Failed to disconnect all users: '.$e->getMessage());
        }
    }

    /**
     * Disconnect by MAC.
     */
    public function kickByMac(Request $request, Router $router)
    {
        $mac = $request->input('mac');
        if (! $mac) {
            return back()->with('error', 'Missing MAC address.');
        }

        try {
            MikroTikHotspotService::disconnectByMac($router, $mac);
            return back()->with('status', 'Sessions with this MAC were disconnected.');
        } catch (\Throwable $e) {
            return back()->with('error', 'Failed to disconnect by MAC: '.$e->getMessage());
        }
    }

    /**
     * Disable hotspot user.
     */
    public function disable(Request $request, Router $router)
    {
        $username = $request->input('user');

        if ($username) {
            try {
                MikroTikHotspotService::disableUser($router, $username);
                return back()->with('status', 'User disabled successfully.');
            } catch (\Throwable $e) {
                return back()->with('error', 'Failed to disable user: '.$e->getMessage());
            }
        }

        return back()->with('error', 'Missing username.');
    }

    /**
     * Export current view to CSV.
     */
    public function export(Request $request): StreamedResponse
    {
        // reuse index logic but without view
        $requestExport = $request->duplicate();
        $data = app(self::class)->index($requestExport)->getData();

        /** @var \App\Models\Router $router */
        $router = $data['currentRouter'];
        $sessions = $data['sessions'] ?? [];

        $filename = 'hotspot-active-'.$router->id.'-'.now()->format('Ymd_His').'.csv';

        $headers = [
            'Content-Type'        => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"$filename\"",
        ];

        return response()->stream(function () use ($sessions) {
            $out = fopen('php://output', 'w');
            fputcsv($out, ['server','user','address','mac','uptime','bytes_in','bytes_out','traffic_bytes','time_left','login_by','comment']);

            foreach ($sessions as $s) {
                fputcsv($out, [
                    $s['server'] ?? $s['server-name'] ?? '',
                    $s['user'] ?? $s['name'] ?? '',
                    $s['address'] ?? '',
                    $s['mac-address'] ?? '',
                    $s['uptime'] ?? '',
                    $s['_bytes_in'] ?? '',
                    $s['_bytes_out'] ?? '',
                    $s['_traffic'] ?? '',
                    $s['session-time-left'] ?? $s['time_left'] ?? '',
                    $s['login-by'] ?? '',
                    $s['comment'] ?? '',
                ]);
            }
            fclose($out);
        }, 200, $headers);
    }

    /**
     * History page (simple).
     */
    public function history(Request $request)
    {
        $routerId = $request->integer('router_id');
        $user     = $request->get('user');

        $query = DB::table('hotspot_session_logs')->orderByDesc('seen_at');

        if ($routerId) {
            $query->where('router_id', $routerId);
        }

        if ($user) {
            $query->where('user', $user);
        }

        $logs = $query->limit(200)->get();

        return view('hotspot.active.history', [
            'logs'  => $logs,
            'user'  => $user,
            'router_id' => $routerId,
        ]);
    }

    /* ---------- helpers ---------- */

    protected static function parseDurationToSeconds(string $str): int
    {
        // format like "20h13m33s" or "05m16s"
        if ($str === '') return 0;

        $pattern = '/(?:(\d+)w)?(?:(\d+)d)?(?:(\d+)h)?(?:(\d+)m)?(?:(\d+)s)?/';
        if (! preg_match($pattern, $str, $m)) {
            return 0;
        }
        $w = (int)($m[1] ?? 0);
        $d = (int)($m[2] ?? 0);
        $h = (int)($m[3] ?? 0);
        $mi= (int)($m[4] ?? 0);
        $s = (int)($m[5] ?? 0);

        return $s + $mi*60 + $h*3600 + $d*86400 + $w*604800;
    }

    protected static function sortSessions(array $sessions, string $sort, string $dir): array
    {
        $keyMap = [
            'traffic'   => '_traffic',
            'bytes_in'  => '_bytes_in',
            'bytes_out' => '_bytes_out',
            'uptime'    => '_uptime_sec',
            'user'      => '_user',
        ];

        $key = $keyMap[$sort] ?? '_traffic';
        $mul = $dir === 'asc' ? 1 : -1;

        usort($sessions, function ($a, $b) use ($key, $mul) {
            $va = $a[$key] ?? 0;
            $vb = $b[$key] ?? 0;
            if ($va == $vb) return 0;
            return ($va < $vb ? -1 : 1) * $mul;
        });

        return $sessions;
    }

    protected function logSnapshot(Router $router, array $sessions): void
    {
        if (empty($sessions)) {
            return;
        }

        $rows = [];
        $now  = now();

        foreach ($sessions as $s) {
            $rows[] = [
                'router_id'  => $router->id,
                'user'       => $s['user'] ?? $s['name'] ?? '',
                'address'    => $s['address'] ?? '',
                'mac'        => $s['mac-address'] ?? '',
                'bytes_in'   => $s['_bytes_in'] ?? 0,
                'bytes_out'  => $s['_bytes_out'] ?? 0,
                'uptime'     => $s['uptime'] ?? '',
                'login_by'   => $s['login-by'] ?? '',
                'seen_at'    => $now,
            ];
        }

        DB::table('hotspot_session_logs')->insert($rows);
    }
}