<?php
// public/thankyou.php

use App\Models\Order;
use App\Services\Hotspot\VoucherService;
use Illuminate\Contracts\Console\Kernel;

define('AZAM_WEBHOOK_LOG', __DIR__ . '/../storage/logs/azam_thankyou_webhook.log');

// 1) If AzamPay is POSTing here, handle the webhook
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $raw = file_get_contents('php://input');

    // --- log RAW body for debugging ---
    file_put_contents(
        AZAM_WEBHOOK_LOG,
        date('c') . " RAW\n" . $raw . "\n\n",
        FILE_APPEND
    );

    $data = json_decode($raw, true);
    if (!is_array($data)) {
        http_response_code(400);
        echo 'Invalid JSON';
        exit;
    }

    // Expected Azam payload example:
    // {
    //   "transactionstatus":"success",
    //   "operator":"Airtel",
    //   "utilityref":"BB2025....",
    //   "amount":"200",
    //   "msisdn":"2557...",
    //   ...
    // }

    $utilityRef = $data['utilityref'] ?? null;            // our order->reference
    $statusStr  = strtolower($data['transactionstatus'] ?? '');
    $msisdn     = $data['msisdn'] ?? null;
    $amount     = (int)($data['amount'] ?? 0);

    // Log parsed payload
    file_put_contents(
        AZAM_WEBHOOK_LOG,
        date('c') . " PARSED " . json_encode([
            'transactionstatus' => $statusStr,
            'utilityref'        => $utilityRef,
            'reference'         => $data['reference'] ?? null,
            'amount'            => $amount,
            'msisdn'            => $msisdn,
        ]) . "\n",
        FILE_APPEND
    );

    if (!$utilityRef) {
        http_response_code(200);
        echo 'Missing utilityref';
        exit;
    }

    // 2) Bootstrap Laravel so we can use Eloquent + VoucherService
    require __DIR__ . '/../vendor/autoload.php';
    $app = require __DIR__ . '/../bootstrap/app.php';

    /** @var Kernel $kernel */
    $kernel = $app->make(Kernel::class);
    $kernel->bootstrap();

    // 3) Find the order by reference == utilityref
    /** @var Order|null $order */
    $order = Order::where('reference', $utilityRef)->first();

    if (!$order) {
        file_put_contents(
            AZAM_WEBHOOK_LOG,
            date('c') . " ORDER_NOT_FOUND " . json_encode(['utilityref' => $utilityRef]) . "\n",
            FILE_APPEND
        );

        \Log::warning('Azam thankyou: order not found', [
            'utilityref' => $utilityRef,
            'payload'    => $data,
        ]);

        http_response_code(200);
        echo 'Order not found';
        exit;
    }

    file_put_contents(
        AZAM_WEBHOOK_LOG,
        date('c') . " ORDER_FOUND " . json_encode([
            'order_id'  => $order->id,
            'reference' => $order->reference,
            'status'    => $order->status,
        ]) . "\n",
        FILE_APPEND
    );

    // Save last callback in meta
    $meta = $order->meta ?? [];
    $meta['azam_last_callback'] = $data;
    $order->meta = $meta;

    // 4) Decide success vs failed (based on transactionstatus)
    $isSuccess = ($statusStr === 'success');

    if ($isSuccess) {
        // ✅ Payment is SUCCESSFUL from Azam viewpoint
        $order->status = 'success';
        $order->save();

        // Only generate voucher once
        if (!$order->voucher_code) {
            try {
                /** @var VoucherService $vouchers */
                $vouchers = $app->make(VoucherService::class);

                $code = $vouchers->generateForOrder($order);

                file_put_contents(
                    AZAM_WEBHOOK_LOG,
                    date('c') . " VOUCHER_OK " . json_encode([
                        'order_id' => $order->id,
                        'code'     => $code,
                    ]) . "\n",
                    FILE_APPEND
                );

                \Log::info('Azam thankyou: voucher generated', [
                    'order_id' => $order->id,
                    'code'     => $code,
                ]);
            } catch (\Throwable $e) {
                // ❗ Voucher failed, but PAYMENT is still success.
                //    Do NOT flip order->status back to failed.
                file_put_contents(
                    AZAM_WEBHOOK_LOG,
                    date('c') . " VOUCHER_ERROR " . json_encode([
                        'order_id' => $order->id,
                        'error'    => $e->getMessage(),
                    ]) . "\n",
                    FILE_APPEND
                );

                \Log::error('Azam thankyou: voucher generation failed', [
                    'order_id' => $order->id,
                    'error'    => $e->getMessage(),
                ]);

                // Keep status as success, just note the error
                $meta = $order->meta ?? [];
                $meta['voucher_error'] = $e->getMessage();
                $order->meta = $meta;
                $order->save();

                http_response_code(200);
                echo 'Payment success, voucher generation failed';
                exit;
            }
        } else {
            // Already has voucher_code – just log and exit
            file_put_contents(
                AZAM_WEBHOOK_LOG,
                date('c') . " VOUCHER_ALREADY_EXISTS " . json_encode([
                    'order_id' => $order->id,
                    'code'     => $order->voucher_code,
                ]) . "\n",
                FILE_APPEND
            );
        }
    } else {
        // ❌ Payment failed
        $order->status = 'failed';
        $order->save();

        file_put_contents(
            AZAM_WEBHOOK_LOG,
            date('c') . " PAYMENT_FAILED " . json_encode([
                'order_id' => $order->id,
                'status'   => $statusStr,
            ]) . "\n",
            FILE_APPEND
        );
    }

    http_response_code(200);
    echo 'OK';
    exit;
}

// 2) If someone opens this in a browser (GET), show a simple page
$transactionId = $_GET['transaction_id'] ?? '';

?>
<!DOCTYPE html>
<html lang="sw">
<head>
    <meta charset="UTF-8">
    <title>Asante kwa malipo</title>
</head>
<body>
    <h1>Asante kwa kutumia huduma yetu ya WiFi.</h1>
    <p>Kama malipo yako yamekamilika, utapokea vocha kwa SMS ndani ya muda mfupi.</p>
</body>
</html>