<?php

namespace App\Services\Payments;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use RuntimeException;

class AzamPayService
{
    /**
     * This class mimics your working plain-php integration:
     *
     * 1) Token: https://authenticator.azampay.co.tz/AppRegistration/GenerateToken
     * 2) Checkout: {azampay_base_url}/azampay/mno/checkout
     *
     * All credentials are loaded from the `settings` table:
     *   azampay_app_name, azampay_client_id, azampay_client_secret, azampay_base_url
     */

    protected string $authUrl = 'https://authenticator.azampay.co.tz/AppRegistration/GenerateToken';

    protected string $checkoutBase;
    protected string $appName;
    protected string $clientId;
    protected string $clientSecret;

    public function __construct()
    {
        // Load from settings table (same as your plain PHP code)
        $settings = DB::table('settings')
            ->whereIn('key', [
                'azampay_app_name',
                'azampay_client_id',
                'azampay_client_secret',
                'azampay_base_url',
            ])
            ->pluck('value', 'key');

        $this->appName      = $settings['azampay_app_name']      ?? '';
        $this->clientId     = $settings['azampay_client_id']     ?? '';
        $this->clientSecret = $settings['azampay_client_secret'] ?? '';
        $this->checkoutBase = rtrim($settings['azampay_base_url'] ?? 'https://checkout.azampay.co.tz', '/');
    }

    /**
     * Request AzamPay access token (same payload as azampay.php).
     *
     * @return string
     * @throws RuntimeException
     */
    protected function getToken(): string
    {
        $payload = [
            'appName'      => $this->appName,
            'clientId'     => $this->clientId,
            'clientSecret' => $this->clientSecret,
        ];

        $resp = Http::asJson()->post($this->authUrl, $payload);

        if (! $resp->successful()) {
            logger()->error('AzamPay token request failed', [
                'status' => $resp->status(),
                'body'   => $resp->body(),
            ]);

            throw new RuntimeException('Failed to get AzamPay token. HTTP '.$resp->status());
        }

        $json  = $resp->json();
        $token = $json['data']['accessToken'] ?? null;

        if (! $token) {
            logger()->error('AzamPay token missing in response', ['json' => $json]);
            throw new RuntimeException('Failed to get AzamPay token (no accessToken).');
        }

        return $token;
    }

    /**
     * Start an MNO checkout (Airtel, Tigo, Halopesa, Azampesa…)
     * using the same structure as your working azampay.php.
     *
     * @param  string  $msisdn      Normalised phone e.g. 2557XXXXXXXX
     * @param  int     $amount      Amount in TZS
     * @param  string  $externalId  Our internal reference / order id
     * @param  string  $provider    Airtel|Tigo|Halopesa|Azampesa
     * @return array   Decoded JSON response from AzamPay
     */
    public function chargeMno(string $msisdn, int $amount, string $externalId, string $provider = 'Airtel'): array
    {
        $token = $this->getToken();

        $url = $this->checkoutBase . '/azampay/mno/checkout';

        $body = [
            'accountNumber' => $msisdn,
            'amount'        => $amount,
            'currency'      => 'TZS',
            'externalId'    => $externalId,
            'provider'      => $provider,
        ];

        $resp = Http::withToken($token)
            ->asJson()
            ->post($url, $body);

        $json = $resp->json();

        if (! $resp->successful()) {
            logger()->error('AzamPay checkout failed', [
                'status'  => $resp->status(),
                'body'    => $resp->body(),
                'request' => $body,
            ]);
        } else {
            logger()->info('AzamPay checkout initiated', [
                'status'  => $resp->status(),
                'body'    => $json,
                'request' => $body,
            ]);
        }

        return $json ?? [];
    }
}

