<?php

namespace App\Services\Payments;

use App\Models\Order;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Log;

class AzamPayGateway
{
    protected Client $http;
    protected string $baseUrl;
    protected string $clientId;
    protected string $clientSecret;
    protected string $account;
    protected string $callbackUrl;

    public function __construct()
    {
        $this->baseUrl      = config('services.azam.base_url');
        $this->clientId     = config('services.azam.client_id');
        $this->clientSecret = config('services.azam.client_secret');
        $this->account      = config('services.azam.account');
        $this->callbackUrl  = config('services.azam.callback_url');

        $this->http = new Client([
            'base_uri' => $this->baseUrl,
            'timeout'  => 10,
            'verify'   => false,
        ]);
    }

    /**
     * Get access token from AzamPay (client_credentials).
     */
    protected function getAccessToken(): ?string
    {
        try {
            $resp = $this->http->post('/auth/token', [
                'json' => [
                    'client_id'     => $this->clientId,
                    'client_secret' => $this->clientSecret,
                    'grant_type'    => 'client_credentials',
                ],
            ]);

            $body = json_decode($resp->getBody()->getContents(), true);
            return $body['access_token'] ?? null;
        } catch (\Throwable $e) {
            Log::error('AzamPay token error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Create AzamPay checkout for an order.
     * Returns redirect URL or null.
     */
    public function createCheckout(Order $order): ?string
    {
        $token = $this->getAccessToken();
        if (!$token) {
            return null;
        }

        // NOTE: Payload structure may need tweaking based on AzamPay docs.
        $payload = [
            'accountNumber'  => $this->account,
            'amount'         => (float)$order->amount,
            'currency'       => $order->currency,
            'externalId'     => $order->reference,
            'customerPhone'  => $order->phone,
            'redirectUrl'    => route('payments.success'),
            'callbackUrl'    => $this->callbackUrl,
            'description'    => 'BlueBando voucher purchase',
        ];

        try {
            $resp = $this->http->post('/checkout/create', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $token,
                    'Accept'        => 'application/json',
                ],
                'json' => $payload,
            ]);

            $body = json_decode($resp->getBody()->getContents(), true);
            $url  = $body['checkoutUrl'] ?? ($body['redirectUrl'] ?? null);

            $order->meta = array_merge($order->meta ?? [], [
                'azam_request'  => $payload,
                'azam_response' => $body,
            ]);
            $order->save();

            return $url;
        } catch (\Throwable $e) {
            Log::error('AzamPay create checkout error: ' . $e->getMessage());
            return null;
        }
    }
}
