<?php
namespace App\Services\Payments;

use Illuminate\Support\Facades\Http;
use App\Models\Order;
use RuntimeException;

class AzamPayAdapter implements PaymentGateway
{
    protected function token(): string
    {
        $tokenUrl = rtrim(env('PAYMENT_AZAM_TOKEN_URL',''), '/');
        if (!$tokenUrl) { throw new RuntimeException('PAYMENT_AZAM_TOKEN_URL not set'); }

        $payload = [
            'appName'       => env('PAYMENT_AZAM_APP_NAME'),
            'clientId'      => env('PAYMENT_AZAM_CLIENT_ID'),
            'clientSecret'  => env('PAYMENT_AZAM_CLIENT_SECRET'),
        ];

        $req = Http::withHeaders([
                'X-API-KEY' => env('PAYMENT_AZAM_X_API_KEY',''),
                'Accept'    => 'application/json'
            ])->asJson()
            ->post($tokenUrl, $payload);

        if (!$req->successful()) {
            throw new RuntimeException('AzamPay token error: '.$req->body());
        }

        $data = $req->json();
        $token = $data['data']['accessToken'] ?? $data['access_token'] ?? null;
        if (!$token) throw new RuntimeException('AzamPay token missing in response');
        return $token;
    }

    public function createCheckout(array $p): PaymentIntent
    {
        $createUrl = rtrim(env('PAYMENT_AZAM_CREATE_URL',''), '/');
        if (!$createUrl) { throw new RuntimeException('PAYMENT_AZAM_CREATE_URL not set'); }

        $token = $this->token();

        $payload = [
            'amount'          => $p['amount'],
            'currency'        => $p['currency'] ?? 'TZS',
            'external_ref'    => $p['reference'],
            'description'     => $p['description'] ?? 'BlueBando Payment',
            'redirect_url'    => $p['return_url'],
            'notification_url'=> $p['webhook_url'],
        ];
        if ($opt = env('PAYMENT_AZAM_PAYMENT_OPTION')) {
            $payload['payment_option'] = $opt;
        }

        $req = Http::withHeaders([
                'Authorization' => 'Bearer '.$token,
                'X-API-KEY'     => env('PAYMENT_AZAM_X_API_KEY',''),
                'Accept'        => 'application/json'
            ])->asJson()
            ->post($createUrl, $payload);

        if (!$req->successful()) {
            throw new RuntimeException('AzamPay create checkout error: '.$req->body());
        }
        $data = $req->json();
        $url  = $data['data']['redirect_url'] ?? $data['redirect_url'] ?? $data['payment_url'] ?? null;
        $id   = $data['data']['order_id'] ?? $data['order_id'] ?? null;

        return new PaymentIntent($id, $url, $p['reference']);
    }

    public function verify(string $reference): PaymentStatus
    {
        return new PaymentStatus('unknown');
    }

    public function handleWebhook(array $request): void
    {
        $ref = $request['external_ref'] ?? $request['reference'] ?? $request['data']['tx_ref'] ?? null;
        if (!$ref) return;
        $order = Order::where('reference', $ref)->first();
        if (!$order) return;

        $status = 'failed';
        $flag = $request['transactionstatus'] ?? $request['status'] ?? ($request['data']['status'] ?? '');
        $flag = is_string($flag) ? strtolower($flag) : $flag;
        if (in_array($flag, ['success','successful','paid','completed'], true)) {
            $status = 'paid';
        }
        $order->update(['status' => $status, 'meta_json' => $request]);
    }
}
