<?php

namespace App\Http\Controllers;

use App\Jobs\PollRouterJob;
use App\Models\Router;
use Illuminate\Http\Request;

class RouterController extends Controller
{
    public function index()
    {
        $q = Router::query();
        if($s = request('s')) $q->where('name','like',"%$s%")->orWhere('host','like',"%$s%");
        $routers = $q->latest()->paginate(15);
        return view('routers.index', compact('routers'));
    }

    public function create(){ return view('routers.create'); }

    public function store(Request $r)
{
    $data = $r->validate([
        'name'           => 'required',
        'connection_via' => 'required|in:public,vpn',
        'wan_host'       => 'nullable|ip',                     // Public IP
        'ddns_host'      => 'nullable|string|max:255',         // e.g. xxxx.sn.mynetname.net
        'api_port'       => 'nullable|integer',
        'username'       => 'required',
        'password'       => 'required',
        'use_tls'        => 'boolean',
        'site'           => 'nullable|string|max:191',
    ]);

    // Choose what the "host" field should be right now
    $host = $data['connection_via'] === 'public'
        ? ($data['ddns_host'] ?: $data['wan_host'])
        : null; // VPN will be used later

    $router = \App\Models\Router::create([
        'name'           => $data['name'],
        'connection_via' => $data['connection_via'],
        'wan_host'       => $data['wan_host'] ?: null,
        'ddns_host'      => $data['ddns_host'] ?: null,
        'host'           => $host ?: ($data['wan_host'] ?? null),
        'api_port'       => $data['api_port'] ?? 8728,
        'username'       => $data['username'],
        'password'       => $data['password'],
        'password_enc'   => encrypt($data['password']),
        'use_tls'        => (bool)($data['use_tls'] ?? 0),
        'site'           => $data['site'] ?? null,
        'status'         => 'offline',
    ]);

    // (optional) dispatch poll job here
    // dispatch(new PollRouterJob($router));

    return redirect()->route('routers.index')->with('status','Router added.');
}

    public function edit(Router $router){ return view('routers.edit', compact('router')); }

   public function update(Request $r, \App\Models\Router $router)
{
    $data = $r->validate([
        'name'           => 'required',
        'connection_via' => 'required|in:public,vpn',
        'wan_host'       => 'nullable|ip',
        'ddns_host'      => 'nullable|string|max:255',
        'api_port'       => 'nullable|integer',
        'username'       => 'required',
        'password'       => 'required',
        'use_tls'        => 'boolean',
        'site'           => 'nullable|string|max:191',
    ]);

    $router->fill([
        'name'           => $data['name'],
        'connection_via' => $data['connection_via'],
        'wan_host'       => $data['wan_host'] ?: null,
        'ddns_host'      => $data['ddns_host'] ?: null,
        'api_port'       => $data['api_port'] ?? 8728,
        'username'       => $data['username'],
        'password'       => $data['password'],
        'password_enc'   => encrypt($data['password']),
        'use_tls'        => (bool)($data['use_tls'] ?? 0),
        'site'           => $data['site'] ?? null,
    ]);

    // Keep "host" aligned to the chosen path for clarity
    if ($router->connection_via === 'public') {
        $router->host = $router->ddns_host ?: $router->wan_host;
    } else {
        $router->host = $router->vpn_ip ?: $router->host; // future
    }

    $router->save();

    return redirect()->route('routers.index')->with('status','Router updated.');
}

    public function destroy(Router $router)
    {
        $router->delete();
        return back()->with('status','Router removed.');
    }

    public function show(Router $router)
    {
        return view('routers.show', compact('router'));
    }
}
