<?php

namespace App\Http\Controllers;

use App\Models\Plan;
use App\Models\Router;
use Illuminate\Http\Request;
use App\Services\Router\MikroTikPlanService;

class PlanController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
{
    $routerId = $request->query('router_id');

    $routers = Router::orderBy('name')->get();

    $plans = Plan::with('router')
        ->when($routerId, fn($q) => $q->where('router_id', $routerId))
        ->orderBy('price')
        ->paginate(25);

    // LIVE profiles from MikroTik (only when a router is selected)
    $liveProfiles = [];
    $liveRouter   = null;

    if ($routerId) {
        $liveRouter = $routers->firstWhere('id', $routerId);
        if ($liveRouter) {
            try {
                $liveProfiles = MikroTikPlanService::listProfilesForRouter($liveRouter);
            } catch (\Throwable $e) {
                $liveProfiles = [];
                // optionally log: logger()->warning('Failed to list Mikrotik profiles', ['error'=>$e->getMessage()]);
            }
        }
    }

    return view('plans.index', [
        'plans'        => $plans,
        'routers'      => $routers,
        'routerId'     => $routerId,
        'liveProfiles' => $liveProfiles,
        'liveRouter'   => $liveRouter,
    ]);
}

    public function create()
    {
        $routers = Router::orderBy('name')->get();

        return view('plans.create', compact('routers'));
    }

    public function store(Request $r)
    {
        $data = $r->validate([
            'router_id'         => 'nullable|exists:routers,id',
            'name'              => 'required|string|max:255',
            'profile'           => 'nullable|string|max:255',
            'price'             => 'required|integer|min:0',
            'currency'          => 'nullable|string|max:10',
            'validity_hours'    => 'required|integer|min:1',
            'rate_limit'        => 'nullable|string|max:255',
            'download_limit_mb' => 'nullable|integer|min:0',
            'upload_limit_mb'   => 'nullable|integer|min:0',
            'is_active'         => 'required|boolean',
        ]);

        // default currency if empty
        if (empty($data['currency'])) {
            $data['currency'] = config('bluebando.currency', 'TZS');
        }

        $plan = Plan::create($data);

        // 🔗 write-through to MikroTik
        try {
            MikroTikPlanService::syncPlanToRouter($plan);
        } catch (\Throwable $e) {
            // we don’t fail the request, just log later if needed
            // logger()->warning('MikroTik plan sync failed', ['error' => $e->getMessage()]);
        }

        return redirect()->route('plans.index')
            ->with('status', 'Plan created and synced to router.');
    }

    public function edit(Plan $plan)
    {
        $routers = Router::orderBy('name')->get();

        return view('plans.edit', compact('plan', 'routers'));
    }

    public function update(Request $r, Plan $plan)
    {
        $data = $r->validate([
            'router_id'         => 'nullable|exists:routers,id',
            'name'              => 'required|string|max:255',
            'profile'           => 'nullable|string|max:255',
            'price'             => 'required|integer|min:0',
            'currency'          => 'nullable|string|max:10',
            'validity_hours'    => 'required|integer|min:1',
            'rate_limit'        => 'nullable|string|max:255',
            'download_limit_mb' => 'nullable|integer|min:0',
            'upload_limit_mb'   => 'nullable|integer|min:0',
            'is_active'         => 'required|boolean',
        ]);

        if (empty($data['currency'])) {
            $data['currency'] = config('bluebando.currency', 'TZS');
        }

        $plan->update($data);

        try {
            MikroTikPlanService::syncPlanToRouter($plan);
        } catch (\Throwable $e) {
            // logger()->warning('MikroTik plan sync failed', ['error' => $e->getMessage()]);
        }

        return redirect()->route('plans.index')
            ->with('status', 'Plan updated and synced to router.');
    }

    public function destroy(Plan $plan)
    {
        try {
            MikroTikPlanService::deletePlanFromRouter($plan);
        } catch (\Throwable $e) {
            // logger()->warning('MikroTik delete plan failed', ['error' => $e->getMessage()]);
        }

        $plan->delete();

        return redirect()->route('plans.index')
            ->with('status','Plan deleted (router profile removed if existed).');
    }
}