<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Plan;
use App\Models\Router;
use App\Services\Payments\AzamPayService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class PaymentController extends Controller
{
    /**
     * Show the external payment page (public/payment.php calls this).
     */
    public function showForm(Request $request)
    {
        // Choose router (by ?router_id= or default first router)
        $router = null;
        if ($request->filled('router_id')) {
            $router = Router::find($request->get('router_id'));
        }
        if (! $router) {
            $router = Router::first();
        }

        // Load plans for that router
        $plans = Plan::query()
            ->where('is_active', 1)
            ->when($router, fn ($q) => $q->where('router_id', $router->id))
            ->orderBy('price')
            ->get();

        $networks = ['Airtel', 'Tigo', 'Halopesa', 'Azampesa'];

        return view('payments.form', compact('router', 'plans', 'networks'));
    }

    /**
     * MSISDN normalisation same as your plain PHP.
     */
    protected function normalizeMsisdn(string $phone): string
    {
        $phone = preg_replace('/[^\d]/', '', $phone);

        if (strlen($phone) === 10 && str_starts_with($phone, '0')) {
            return '255' . substr($phone, 1);
        }

        if (strlen($phone) === 13 && str_starts_with($phone, '+255')) {
            return '255' . substr($phone, 4);
        }

        if (strlen($phone) === 12 && str_starts_with($phone, '255')) {
            return $phone;
        }

        if (strlen($phone) === 9 && str_starts_with($phone, '7')) {
            return '255' . $phone;
        }

        throw ValidationException::withMessages([
            'phone' => 'Namba ya simu si sahihi. Tafadhali andika namba kamili mfano 0745XXXXXXX au 745XXXXXXX.',
        ]);
    }

    /**
     * Create order + call AzamPay (MNO checkout).
     * This replaces the old createOrderMinimal() flow.
     */
    public function createOrder(Request $request, AzamPayService $azam)
    {
        $data = $request->validate([
            'router_id' => 'required|exists:routers,id',
            'plan_id'   => 'required|exists:plans,id',
            'phone'     => 'required|string',
            'network'   => 'nullable|string',      // e.g. Airtel, Tigo
            'gateway'   => 'required|in:azampay',  // for now we only support AzamPay here
            'provider'  => 'nullable|string',      // same as network for AzamPay
        ]);

        $router   = Router::findOrFail($data['router_id']);
        $plan     = Plan::findOrFail($data['plan_id']);
        $msisdn   = $this->normalizeMsisdn($data['phone']);
        $provider = $data['provider'] ?? $data['network'] ?? 'Airtel';

        // 1) Create local order (similar to "payments" insert in your plain PHP)
        $order              = new Order();
        $order->router_id   = $router->id;
        $order->plan_id     = $plan->id;
        $order->phone       = $msisdn;
        $order->network     = $provider;
        $order->gateway     = 'azampay';
        $order->amount      = $plan->price;
        $order->currency    = $plan->currency ?? 'TZS';
        $order->reference   = 'BB' . now()->format('YmdHis') . Str::random(4);
        $order->status      = 'pending';
        $order->voucher_code = null;
        $order->meta        = [];
        $order->save();

        // 2) Call AzamPay using same logic as api/gateways/azampay.php
        $azamResponse = $azam->chargeMno(
            $msisdn,
            (int) $order->amount,
            $order->reference,
            $provider
        );

        // 3) Store raw Azam response for debugging
        $meta = $order->meta ?? [];
        $meta['azam_response'] = $azamResponse;
        $order->meta = $meta;
        $order->save();

        // 4) Respond to frontend
        return response()->json([
            'success'   => true,
            'order_id'  => $order->id,
            'reference' => $order->reference,
            'message'   => 'Payment initiated, tafadhali weka PIN kwenye simu yako kuthibitisha.',
        ]);
    }
}
